@echo off
:: -------------------------------------------------------------------------
:: vm_recovery.bat for Nutanix Disaster Recovery
:: -------------------------------------------------------------------------
:: Purpose:
::     This batch file initiates a PowerShell script to check and update
::     the DNS configuration based on the current IP address. All activities
::     are logged to timestamped log files to preserve execution history.
::
:: Description:
::     - Creates log directory if it doesn't exist
::     - Generates timestamped batch execution log (vm_recovery_batch_YYYYMMDD-HHMM.log)
::     - Calls the PowerShell script (Set-DnsConfiguration.ps1) which creates
::       its own detailed log file (Set-DnsConfiguration_YYYYMMDD-HHMMSS.log)
::     - Captures and logs PowerShell script exit code
::     - Each execution creates new log files to prevent overwrites
::
:: Requirements:
::     - Must run with administrative privileges.
::     - Requires PowerShell to be installed.
::
:: Version:
::     2.0
::
:: Author:
::     Mike Dent (mike@mikedent.io)
:: -------------------------------------------------------------------------

set "scriptDir=%~dp0"
set "logDir=C:\Program Files\Nutanix\scripts"

REM Create the log directory if it doesn't exist
if not exist "%logDir%" (
    mkdir "%logDir%"
    if errorlevel 1 (
        echo ERROR: Failed to create log directory: %logDir%
        exit /b 1
    )
)

REM Generate timestamp for batch log
for /f "tokens=2-4 delims=/ " %%a in ('date /t') do set "mydate=%%c%%a%%b"
for /f "tokens=1-2 delims=: " %%a in ('time /t') do set "mytime=%%a%%b"
set "batchLog=%logDir%\vm_recovery_batch_%mydate%-%mytime%.log"

REM Log batch file execution start
echo ============================================================= >> "%batchLog%"
echo vm_recovery.bat - Execution Started >> "%batchLog%"
echo ============================================================= >> "%batchLog%"
echo Date/Time: %date% %time% >> "%batchLog%"
echo Script Directory: %scriptDir% >> "%batchLog%"
echo Log Directory: %logDir% >> "%batchLog%"
echo. >> "%batchLog%"

REM =========================================================================
REM SCRIPT EXECUTION SECTION
REM =========================================================================
REM Add additional PowerShell scripts to execute below
REM Each script will be called in sequence with error checking
REM =========================================================================

set "overallExitCode=0"

REM Script 1: DNS Configuration
echo ------------------------------------------------------------- >> "%batchLog%"
echo Script 1: Set-DnsConfiguration.ps1 >> "%batchLog%"
echo ------------------------------------------------------------- >> "%batchLog%"
powershell -ExecutionPolicy RemoteSigned -File "%scriptDir%Set-DnsConfiguration.ps1" -LogDir "%logDir%"
set "scriptExitCode=%errorlevel%"
if %scriptExitCode% equ 0 (
    echo   Result: SUCCESS ^(Exit Code: %scriptExitCode%^) >> "%batchLog%"
) else (
    echo   Result: FAILURE ^(Exit Code: %scriptExitCode%^) >> "%batchLog%"
    set "overallExitCode=%scriptExitCode%"
)
echo. >> "%batchLog%"

REM =========================================================================
REM ADD ADDITIONAL SCRIPTS HERE
REM =========================================================================
REM Example:
REM echo ------------------------------------------------------------- >> "%batchLog%"
REM echo Script 2: Configure-NetworkSettings.ps1 >> "%batchLog%"
REM echo ------------------------------------------------------------- >> "%batchLog%"
REM powershell -ExecutionPolicy RemoteSigned -File "%scriptDir%Configure-NetworkSettings.ps1" -LogDir "%logDir%"
REM set "scriptExitCode=%errorlevel%"
REM if %scriptExitCode% equ 0 (
REM     echo   Result: SUCCESS ^(Exit Code: %scriptExitCode%^) >> "%batchLog%"
REM ) else (
REM     echo   Result: FAILURE ^(Exit Code: %scriptExitCode%^) >> "%batchLog%"
REM     set "overallExitCode=%scriptExitCode%"
REM )
REM echo. >> "%batchLog%"
REM =========================================================================

REM Log completion
echo ============================================================= >> "%batchLog%"
echo vm_recovery.bat - Execution Completed >> "%batchLog%"
if %overallExitCode% equ 0 (
    echo Overall Status: SUCCESS >> "%batchLog%"
) else (
    echo Overall Status: FAILURE ^(Exit Code: %overallExitCode%^) >> "%batchLog%"
)
echo ============================================================= >> "%batchLog%"

REM Exit with overall exit code (first failure encountered)
exit /b %overallExitCode%
