<#
.SYNOPSIS
    DNS and IP Configuration Checker Script

.DESCRIPTION
    This script checks the current IP address against predefined subnets for Site A and Site B. If the IP address
    matches a subnet and the DNS servers are different from the defined configuration, it updates the DNS servers
    accordingly.

.VERSION
    2.0

.AUTHOR
    Mike Dent
    mike@mikedent.io

.NOTES
    - Called from vm_recovery.bat
    - Logs all actions with timestamped log files in "C:\Program Files\Nutanix\scripts\"
    - Log files are named: Set-DnsConfiguration_YYYYMMDD-HHMMSS.log
    - Each execution creates a new log file to preserve history

.PARAMETER LogDir
    Directory to save the log file.
#>

param (
    [string]$LogDir
)

# Create log directory if it doesn't exist
if (-not (Test-Path -Path $LogDir)) {
    try {
        New-Item -Path $LogDir -ItemType Directory -Force | Out-Null
    }
    catch {
        Write-Error "Failed to create log directory: $LogDir"
        exit 1
    }
}

# Generate timestamped log filename
$timestamp = Get-Date -Format "yyyyMMdd-HHmmss"
$logFileName = "Set-DnsConfiguration_$timestamp.log"
$LogFile = Join-Path -Path $LogDir -ChildPath $logFileName

# Function to write log entries
function Write-Log {
    param ([string]$Message)
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $entry = "$timestamp - $Message"

    try {
        Add-Content -Path $LogFile -Value $entry -ErrorAction Stop
    }
    catch {
        Write-Error "Failed to write to log file: $_"
    }
}

# Start logging
Write-Log "-------------------------------------------------------------"
Write-Log "DNS and IP Configuration Checker - Version 2.0"
Write-Log "Script initiated to verify and update DNS settings if needed"
Write-Log "Log File: $LogFile"
Write-Log "-------------------------------------------------------------"

# Define subnet and DNS information for Site A and Site B
# For the Subnets, include multiple in the format of @("subnet1", "subnet2")
# For the DNSServers, include multiple in the format of @("dns1", "dns2")
# Example: @("172.20.11.254", "172.20.12.254")
$sites = @{
    "SiteA" = @{
        "Subnets" = @("172.20.21.0/24")
        "DNSServers" = @("172.20.11.254")
    }
    "SiteB" = @{
        "Subnets" = @("172.20.121.0/24")
        "DNSServers" = @("172.20.14.254")
    }
}

# Function to check if an IP belongs to a subnet using proper CIDR calculation
function Test-Subnet {
    param (
        [string]$IP,
        [string]$Subnet
    )

    try {
        $ipAddress = [System.Net.IPAddress]::Parse($IP)
        $subnetAddress, $prefixLength = $Subnet -split '/'
        $subnetIP = [System.Net.IPAddress]::Parse($subnetAddress)
        $maskBits = [int]$prefixLength

        # Convert IP addresses to binary
        $ipBytes = $ipAddress.GetAddressBytes()
        $subnetBytes = $subnetIP.GetAddressBytes()

        # Create subnet mask
        $maskBytes = [byte[]]::new(4)
        for ($i = 0; $i -lt 4; $i++) {
            $bitsInByte = [Math]::Min($maskBits - ($i * 8), 8)
            if ($bitsInByte -gt 0) {
                $maskBytes[$i] = [byte](0xFF -shl (8 - $bitsInByte))
            }
        }

        # Compare network portions
        for ($i = 0; $i -lt 4; $i++) {
            if (($ipBytes[$i] -band $maskBytes[$i]) -ne ($subnetBytes[$i] -band $maskBytes[$i])) {
                return $false
            }
        }

        return $true
    }
    catch {
        Write-Log "Error in Test-Subnet: $_"
        return $false
    }
}

# Get the primary active network adapter (sorted by InterfaceMetric for priority)
$activeAdapter = Get-NetAdapter |
    Where-Object {$_.Status -eq "Up"} |
    Sort-Object -Property InterfaceMetric |
    Select-Object -First 1

if (-not $activeAdapter) {
    Write-Log "Error: No active network adapter found. Exiting."
    exit 1
}

$interfaceIndex = $activeAdapter.InterfaceIndex
$interfaceAlias = $activeAdapter.InterfaceAlias

Write-Log "Primary Network Adapter: $interfaceAlias (Index: $interfaceIndex, Metric: $($activeAdapter.InterfaceMetric))"

# Get the current IP address from the primary adapter only
$currentIPConfig = Get-NetIPAddress -InterfaceIndex $interfaceIndex -AddressFamily IPv4 |
    Where-Object {$_.IPAddress -ne "127.0.0.1"} |
    Select-Object -First 1

if (-not $currentIPConfig) {
    Write-Log "Error: No valid IPv4 address found on adapter $interfaceAlias. Exiting."
    exit 1
}

$currentIP = $currentIPConfig.IPAddress

# Get current DNS servers from the primary adapter only
$currentDNSConfig = Get-DnsClientServerAddress -InterfaceIndex $interfaceIndex -AddressFamily IPv4
$currentDNS = $currentDNSConfig.ServerAddresses | Where-Object { $_ -ne $null -and $_ -ne "" }

Write-Log "Current IP Address: $currentIP"
Write-Log "Current DNS Servers: $($currentDNS -join ', ')"

# Store previous and new DNS values for logging
$previousDNS = $currentDNS
$newDNS = $null

# Determine site based on current IP and update DNS if necessary
$siteMatched = $false

foreach ($site in $sites.Keys) {
    $siteData = $sites[$site]

    # Check all subnets for this site
    foreach ($subnet in $siteData["Subnets"]) {
        if (Test-Subnet -IP $currentIP -Subnet $subnet) {
            Write-Log "Matched $currentIP to subnet $subnet in $site"
            $siteMatched = $true

            # Compare DNS server settings (order-independent)
            $desiredDNS = $siteData["DNSServers"]

            # Check if DNS arrays are equivalent (same servers, any order)
            $dnsMatch = $true
            if ($null -eq $currentDNS -or $currentDNS.Count -ne $desiredDNS.Count) {
                $dnsMatch = $false
            }
            else {
                foreach ($dns in $desiredDNS) {
                    if ($dns -notin $currentDNS) {
                        $dnsMatch = $false
                        break
                    }
                }
            }

            if ($dnsMatch) {
                Write-Log "DNS servers are already correctly configured for $site. No changes needed."
                $newDNS = $currentDNS
            }
            else {
                Write-Log "DNS mismatch detected."
                Write-Log "  Expected: $($desiredDNS -join ', ')"
                Write-Log "  Current:  $($currentDNS -join ', ')"
                Write-Log "Updating DNS servers for $site..."

                try {
                    # Set new DNS servers directly (no need to clear first)
                    Set-DnsClientServerAddress -InterfaceIndex $interfaceIndex -ServerAddresses $desiredDNS -ErrorAction Stop
                    Write-Log "DNS servers updated successfully to: $($desiredDNS -join ', ')"
                    $newDNS = $desiredDNS

                    # Verify the DNS settings after update
                    Start-Sleep -Milliseconds 500  # Brief pause for changes to take effect
                    $verifyDNS = (Get-DnsClientServerAddress -InterfaceIndex $interfaceIndex -AddressFamily IPv4).ServerAddresses

                    $verifyMatch = $true
                    if ($verifyDNS.Count -ne $desiredDNS.Count) {
                        $verifyMatch = $false
                    }
                    else {
                        foreach ($dns in $desiredDNS) {
                            if ($dns -notin $verifyDNS) {
                                $verifyMatch = $false
                                break
                            }
                        }
                    }

                    if ($verifyMatch) {
                        Write-Log "DNS update verification: PASSED"
                    }
                    else {
                        Write-Log "WARNING: DNS update verification FAILED"
                        Write-Log "  Expected: $($desiredDNS -join ', ')"
                        Write-Log "  Actual:   $($verifyDNS -join ', ')"
                    }
                }
                catch {
                    Write-Log "ERROR: Failed to update DNS servers - $_"
                    $newDNS = $currentDNS
                }
            }

            break
        }
    }

    if ($siteMatched) {
        break
    }
}

if (-not $siteMatched) {
    Write-Log "WARNING: Current IP $currentIP does not match any configured site subnets. No DNS changes made."
    $newDNS = $currentDNS
}

# Summary table in log file
Write-Log "-------------------------------------------------------------"
Write-Log "SUMMARY"
Write-Log "-------------------------------------------------------------"
Write-Log "Adapter:          $interfaceAlias"
Write-Log "IP Address:       $currentIP"
Write-Log "Previous DNS:     $($previousDNS -join ', ')"
Write-Log "Current DNS:      $($newDNS -join ', ')"
Write-Log "Changes Made:     $(if ($previousDNS -join ',' -eq $newDNS -join ',') { 'No' } else { 'Yes' })"
Write-Log "Log File:         $LogFile"
Write-Log "-------------------------------------------------------------"
Write-Log "Script completed successfully."
